/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.http;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;
import java.util.function.Supplier;
import software.amazon.awssdk.annotations.SdkProtectedApi;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.utils.FunctionalUtils;
import software.amazon.awssdk.utils.IoUtils;
import software.amazon.awssdk.utils.StringInputStream;
import software.amazon.awssdk.utils.Validate;
import software.amazon.awssdk.utils.internal.EnumUtils;

@FunctionalInterface
@SdkPublicApi
public interface ContentStreamProvider {
    public static ContentStreamProvider fromByteArray(byte[] bytes) {
        Validate.paramNotNull(bytes, "bytes");
        byte[] copy = Arrays.copyOf(bytes, bytes.length);
        return ContentStreamProvider.fromByteArrayUnsafe(copy);
    }

    public static ContentStreamProvider fromByteArrayUnsafe(final byte[] bytes) {
        Validate.paramNotNull(bytes, "bytes");
        return new ContentStreamProvider(){

            @Override
            public InputStream newStream() {
                return new ByteArrayInputStream(bytes);
            }

            @Override
            public String name() {
                return ProviderType.BYTES.getName();
            }
        };
    }

    public static ContentStreamProvider fromString(final String string, final Charset charset) {
        Validate.paramNotNull(string, "string");
        Validate.paramNotNull(charset, "charset");
        return new ContentStreamProvider(){

            @Override
            public InputStream newStream() {
                return new StringInputStream(string, charset);
            }

            @Override
            public String name() {
                return ProviderType.STRING.getName();
            }
        };
    }

    public static ContentStreamProvider fromUtf8String(String string) {
        return ContentStreamProvider.fromString(string, StandardCharsets.UTF_8);
    }

    public static ContentStreamProvider fromInputStream(final InputStream inputStream) {
        Validate.paramNotNull(inputStream, "inputStream");
        IoUtils.markStreamWithMaxReadLimit(inputStream);
        return new ContentStreamProvider(){
            private boolean first = true;

            @Override
            public InputStream newStream() {
                if (this.first) {
                    this.first = false;
                    return inputStream;
                }
                if (inputStream.markSupported()) {
                    FunctionalUtils.invokeSafely(inputStream::reset);
                    return inputStream;
                }
                throw new IllegalStateException("Content input stream does not support mark/reset, and was already read once.");
            }

            @Override
            public String name() {
                return ProviderType.STREAM.getName();
            }
        };
    }

    public static ContentStreamProvider fromInputStreamSupplier(final Supplier<InputStream> inputStreamSupplier) {
        Validate.paramNotNull(inputStreamSupplier, "inputStreamSupplier");
        return new ContentStreamProvider(){
            private InputStream lastStream;

            @Override
            public InputStream newStream() {
                if (this.lastStream != null) {
                    FunctionalUtils.invokeSafely(this.lastStream::close);
                }
                this.lastStream = (InputStream)inputStreamSupplier.get();
                return this.lastStream;
            }

            @Override
            public String name() {
                return ProviderType.STREAM.getName();
            }
        };
    }

    public InputStream newStream();

    default public String name() {
        return ProviderType.UNKNOWN.getName();
    }

    @SdkProtectedApi
    public static enum ProviderType {
        FILE("File", "f"),
        BYTES("Bytes", "b"),
        STRING("String", "c"),
        STREAM("Stream", "s"),
        UNKNOWN("Unknown", "u");

        private static final Map<String, ProviderType> VALUE_MAP;
        private final String name;
        private final String shortValue;

        private ProviderType(String name, String shortValue) {
            this.name = name;
            this.shortValue = shortValue;
        }

        public String getName() {
            return this.name;
        }

        public String getShortValue() {
            return this.shortValue;
        }

        public static String shortValueFromName(String name) {
            return VALUE_MAP.getOrDefault(name, UNKNOWN).getShortValue();
        }

        static {
            VALUE_MAP = EnumUtils.uniqueIndex(ProviderType.class, ProviderType::getName);
        }
    }
}

