/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.http.auth.aws.internal.signer;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import org.reactivestreams.Publisher;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.checksums.SdkChecksum;
import software.amazon.awssdk.checksums.spi.ChecksumAlgorithm;
import software.amazon.awssdk.http.ContentStreamProvider;
import software.amazon.awssdk.http.SdkHttpRequest;
import software.amazon.awssdk.http.auth.aws.internal.signer.CredentialScope;
import software.amazon.awssdk.http.auth.aws.internal.signer.NoOpPayloadChecksumStore;
import software.amazon.awssdk.http.auth.aws.internal.signer.RollingSigner;
import software.amazon.awssdk.http.auth.aws.internal.signer.V4PayloadSigner;
import software.amazon.awssdk.http.auth.aws.internal.signer.V4RequestSigningResult;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.AsyncChunkEncodedPayload;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.ChecksumTrailerProvider;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.ChunkedEncodedInputStream;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.ChunkedEncodedPayload;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.ChunkedEncodedPublisher;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.SigV4ChunkExtensionProvider;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.SigV4TrailerProvider;
import software.amazon.awssdk.http.auth.aws.internal.signer.chunkedencoding.SyncChunkEncodedPayload;
import software.amazon.awssdk.http.auth.aws.internal.signer.io.ResettableContentStreamProvider;
import software.amazon.awssdk.http.auth.aws.internal.signer.util.ChecksumUtil;
import software.amazon.awssdk.http.auth.aws.internal.signer.util.SignerUtils;
import software.amazon.awssdk.http.auth.spi.signer.PayloadChecksumStore;
import software.amazon.awssdk.utils.BinaryUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.Pair;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public final class AwsChunkedV4PayloadSigner
implements V4PayloadSigner {
    private static final Logger LOG = Logger.loggerFor(AwsChunkedV4PayloadSigner.class);
    private final CredentialScope credentialScope;
    private final int chunkSize;
    private final ChecksumAlgorithm checksumAlgorithm;
    private final PayloadChecksumStore payloadChecksumStore;
    private final List<Pair<String, List<String>>> preExistingTrailers = new ArrayList<Pair<String, List<String>>>();

    private AwsChunkedV4PayloadSigner(Builder builder) {
        this.credentialScope = Validate.paramNotNull(builder.credentialScope, "CredentialScope");
        this.chunkSize = Validate.isPositive(builder.chunkSize, "ChunkSize");
        this.checksumAlgorithm = builder.checksumAlgorithm;
        this.payloadChecksumStore = builder.checksumStore == null ? NoOpPayloadChecksumStore.create() : builder.checksumStore;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public ContentStreamProvider sign(ContentStreamProvider payload, V4RequestSigningResult requestSigningResult) {
        ChunkedEncodedInputStream.Builder chunkedEncodedInputStreamBuilder = ChunkedEncodedInputStream.builder().inputStream(payload.newStream()).chunkSize(this.chunkSize).header(chunk -> Integer.toHexString(chunk.remaining()).getBytes(StandardCharsets.UTF_8));
        SyncChunkEncodedPayload chunkedPayload = new SyncChunkEncodedPayload(chunkedEncodedInputStreamBuilder);
        this.signCommon(chunkedPayload, requestSigningResult);
        return new ResettableContentStreamProvider(chunkedEncodedInputStreamBuilder::build);
    }

    @Override
    public Publisher<ByteBuffer> signAsync(Publisher<ByteBuffer> payload, V4RequestSigningResult requestSigningResult) {
        ChunkedEncodedPublisher.Builder chunkedStreamBuilder = ChunkedEncodedPublisher.builder().publisher(payload).chunkSize(this.chunkSize).addEmptyTrailingChunk(true);
        AsyncChunkEncodedPayload chunkedPayload = new AsyncChunkEncodedPayload(chunkedStreamBuilder);
        this.signCommon(chunkedPayload, requestSigningResult);
        return chunkedStreamBuilder.build();
    }

    private void signCommon(ChunkedEncodedPayload payload, V4RequestSigningResult requestSigningResult) {
        String checksum;
        this.preExistingTrailers.forEach(t -> payload.addTrailer(() -> t));
        SdkHttpRequest.Builder request = requestSigningResult.getSignedRequest();
        payload.decodedContentLength(request.firstMatchingHeader("x-amz-decoded-content-length").map(Long::parseLong).orElseThrow(() -> {
            String msg = String.format("Expected header '%s' to be present", "x-amz-decoded-content-length");
            return new RuntimeException(msg);
        }));
        switch (checksum = request.firstMatchingHeader("x-amz-content-sha256").orElseThrow(() -> new IllegalArgumentException("x-amz-content-sha256 must be set!"))) {
            case "STREAMING-AWS4-HMAC-SHA256-PAYLOAD": {
                RollingSigner rollingSigner = new RollingSigner(requestSigningResult.getSigningKey(), requestSigningResult.getSignature());
                payload.addExtension(new SigV4ChunkExtensionProvider(rollingSigner, this.credentialScope));
                break;
            }
            case "STREAMING-UNSIGNED-PAYLOAD-TRAILER": {
                this.setupChecksumTrailerIfNeeded(payload);
                break;
            }
            case "STREAMING-AWS4-HMAC-SHA256-PAYLOAD-TRAILER": {
                this.setupChecksumTrailerIfNeeded(payload);
                RollingSigner rollingSigner = new RollingSigner(requestSigningResult.getSigningKey(), requestSigningResult.getSignature());
                payload.addExtension(new SigV4ChunkExtensionProvider(rollingSigner, this.credentialScope));
                payload.addTrailer(new SigV4TrailerProvider(payload.trailers(), rollingSigner, this.credentialScope));
                break;
            }
            default: {
                throw new UnsupportedOperationException();
            }
        }
    }

    @Override
    public void beforeSigning(SdkHttpRequest.Builder request, ContentStreamProvider payload) {
        long encodedContentLength = 0L;
        long contentLength = SignerUtils.computeAndMoveContentLength(request, payload);
        this.setupPreExistingTrailers(request);
        encodedContentLength = this.calculateEncodedContentLength(request, contentLength);
        if (this.checksumAlgorithm != null) {
            String checksumHeaderName = ChecksumUtil.checksumHeaderName(this.checksumAlgorithm);
            request.appendHeader("x-amz-trailer", checksumHeaderName);
        }
        request.putHeader("Content-Length", Long.toString(encodedContentLength));
        request.appendHeader("Content-Encoding", "aws-chunked");
    }

    @Override
    public CompletableFuture<Pair<SdkHttpRequest.Builder, Optional<Publisher<ByteBuffer>>>> beforeSigningAsync(SdkHttpRequest.Builder request, Publisher<ByteBuffer> payload) {
        return SignerUtils.moveContentLength(request, payload).thenApply(p -> {
            SdkHttpRequest.Builder requestBuilder = (SdkHttpRequest.Builder)p.left();
            this.setupPreExistingTrailers(requestBuilder);
            long decodedContentLength = requestBuilder.firstMatchingHeader("x-amz-decoded-content-length").map(Long::parseLong).orElseThrow(() -> new RuntimeException("x-amz-decoded-content-length header not present"));
            long encodedContentLength = this.calculateEncodedContentLength(request, decodedContentLength);
            if (this.checksumAlgorithm != null) {
                String checksumHeaderName = ChecksumUtil.checksumHeaderName(this.checksumAlgorithm);
                request.appendHeader("x-amz-trailer", checksumHeaderName);
            }
            request.putHeader("Content-Length", Long.toString(encodedContentLength));
            request.appendHeader("Content-Encoding", "aws-chunked");
            return Pair.of(requestBuilder, (Optional)p.right());
        });
    }

    private long calculateEncodedContentLength(SdkHttpRequest.Builder requestBuilder, long decodedContentLength) {
        String checksum;
        long encodedContentLength = 0L;
        encodedContentLength += this.calculateExistingTrailersLength();
        switch (checksum = requestBuilder.firstMatchingHeader("x-amz-content-sha256").orElseThrow(() -> new IllegalArgumentException("x-amz-content-sha256 must be set!"))) {
            case "STREAMING-AWS4-HMAC-SHA256-PAYLOAD": {
                long extensionsLength = 81L;
                encodedContentLength += this.calculateChunksLength(decodedContentLength, extensionsLength);
                break;
            }
            case "STREAMING-UNSIGNED-PAYLOAD-TRAILER": {
                if (this.checksumAlgorithm != null) {
                    encodedContentLength += this.calculateChecksumTrailerLength(ChecksumUtil.checksumHeaderName(this.checksumAlgorithm));
                }
                encodedContentLength += this.calculateChunksLength(decodedContentLength, 0L);
                break;
            }
            case "STREAMING-AWS4-HMAC-SHA256-PAYLOAD-TRAILER": {
                long extensionsLength = 81L;
                encodedContentLength += this.calculateChunksLength(decodedContentLength, extensionsLength);
                if (this.checksumAlgorithm != null) {
                    encodedContentLength += this.calculateChecksumTrailerLength(ChecksumUtil.checksumHeaderName(this.checksumAlgorithm));
                }
                encodedContentLength += 90L;
                break;
            }
            default: {
                throw new UnsupportedOperationException();
            }
        }
        return encodedContentLength += 2L;
    }

    private void setupPreExistingTrailers(SdkHttpRequest.Builder request) {
        for (String header : request.matchingHeaders("x-amz-trailer")) {
            List<String> values = request.matchingHeaders(header);
            if (values.isEmpty()) {
                throw new IllegalArgumentException(header + " must be present in the request headers to be a valid trailer.");
            }
            this.preExistingTrailers.add(Pair.of(header, values));
            request.removeHeader(header);
        }
    }

    private long calculateChunksLength(long contentLength, long extensionsLength) {
        long lengthInBytes = 0L;
        long chunkHeaderLength = Integer.toHexString(this.chunkSize).length();
        long numChunks = contentLength / (long)this.chunkSize;
        lengthInBytes += numChunks * (chunkHeaderLength + extensionsLength + 2L + (long)this.chunkSize + 2L);
        long remainingBytes = contentLength % (long)this.chunkSize;
        if (remainingBytes > 0L) {
            long remainingChunkHeaderLength = Long.toHexString(remainingBytes).length();
            lengthInBytes += remainingChunkHeaderLength + extensionsLength + 2L + remainingBytes + 2L;
        }
        return lengthInBytes += 1L + extensionsLength + 2L;
    }

    private long calculateExistingTrailersLength() {
        long lengthInBytes = 0L;
        for (Pair<String, List<String>> trailer : this.preExistingTrailers) {
            lengthInBytes += this.calculateTrailerLength(trailer);
        }
        return lengthInBytes;
    }

    private long calculateTrailerLength(Pair<String, List<String>> trailer) {
        long lengthInBytes = (long)trailer.left().length() + 1L;
        for (String value : trailer.right()) {
            lengthInBytes += (long)value.length();
        }
        return (lengthInBytes += (long)(trailer.right().size() - 1)) + 2L;
    }

    private long calculateChecksumTrailerLength(String checksumHeaderName) {
        long lengthInBytes = (long)checksumHeaderName.length() + 1L;
        SdkChecksum sdkChecksum = ChecksumUtil.fromChecksumAlgorithm(this.checksumAlgorithm);
        return (lengthInBytes += (long)BinaryUtils.toBase64(sdkChecksum.getChecksumBytes()).length()) + 2L;
    }

    private void setupChecksumTrailerIfNeeded(ChunkedEncodedPayload payload) {
        if (this.checksumAlgorithm == null) {
            return;
        }
        String checksumHeaderName = ChecksumUtil.checksumHeaderName(this.checksumAlgorithm);
        String cachedChecksum = this.getCachedChecksum();
        if (cachedChecksum != null) {
            LOG.debug(() -> String.format("Cached payload checksum available for algorithm %s: %s. Using cached value", this.checksumAlgorithm.algorithmId(), checksumHeaderName));
            payload.addTrailer(() -> Pair.of(checksumHeaderName, Collections.singletonList(cachedChecksum)));
            return;
        }
        SdkChecksum sdkChecksum = ChecksumUtil.fromChecksumAlgorithm(this.checksumAlgorithm);
        ChecksumTrailerProvider checksumTrailer = new ChecksumTrailerProvider(sdkChecksum, checksumHeaderName, this.checksumAlgorithm, this.payloadChecksumStore);
        payload.checksumPayload(sdkChecksum);
        payload.addTrailer(checksumTrailer);
    }

    private String getCachedChecksum() {
        byte[] checksumBytes = this.payloadChecksumStore.getChecksumValue(this.checksumAlgorithm);
        if (checksumBytes != null) {
            return BinaryUtils.toBase64(checksumBytes);
        }
        return null;
    }

    static class Builder {
        private CredentialScope credentialScope;
        private Integer chunkSize;
        private ChecksumAlgorithm checksumAlgorithm;
        private PayloadChecksumStore checksumStore;

        Builder() {
        }

        public Builder credentialScope(CredentialScope credentialScope) {
            this.credentialScope = credentialScope;
            return this;
        }

        public Builder chunkSize(Integer chunkSize) {
            this.chunkSize = chunkSize;
            return this;
        }

        public Builder checksumAlgorithm(ChecksumAlgorithm checksumAlgorithm) {
            this.checksumAlgorithm = checksumAlgorithm;
            return this;
        }

        public Builder checksumStore(PayloadChecksumStore checksumStore) {
            this.checksumStore = checksumStore;
            return this;
        }

        public AwsChunkedV4PayloadSigner build() {
            return new AwsChunkedV4PayloadSigner(this);
        }
    }
}

